#define productionVersion 1

/* *************************************************************************** */
/*																			   */
/* Font Manager Sample Program												   */
/*																			   */
/* Description: Produces and displays catalog of fonts on disk, shows 		   */
/*				font strike of selected font.								   */
/*                                                                             */
/* Developer Technical Support Apple II Sample Code                            */
/*                                                                             */
/* By: Dan Strnad															   */
/*																			   */
/*	   Version 3.0															   */
/*																			   */
/* ShowFont Procedure adapted from HodgePodge example program			 	   */
/*																			   */
/* Copyright 1989-1990 Apple Computer, Inc.									   */
/*																			   */
/* *************************************************************************** */

/* **************************************************************************** */
/*																			    */
/* General Remarks							 								    */
/*																			    */
/* Most applications need not use each of the Font Manager calls demonstrated   */
/* in this program. Besides starting up and shutting down the font manager,     */																		 
/* ChooseFont and GetFamNum are the calls most programmers may find use for.    */																		 
/* QuickDraw II also provides routines and documentation for both basic and     */
/* advanced handling of fonts.													*/
/*																			    */
/* Because memory management strategies differ across applications, no special  */
/* provision is made in this program for handling memory error conditions such  */
/* as an unable to allocate memory block error ($0201). This or other memory    */
/* manager errors can be returned by several of the font manager calls. This 	*/
/* sample program is incomplete with regard to handling of these errors, which  */
/* will cause an error message to be displayed and the program to abort.		*/
/* Commercial applications should differ from this sample program by more		*/
/* carefully handling potentially tight memory conditions.						*/
/*																			    */
/* Also to do with memory management, when fonts are to be marked purgeable 	*/
/* using SetPurgeStat, the id of the font to purge should be that returned from */
/* a font manager call, such as FMGetCurFID, FindFontStats, or LoadFont.		*/
/*																			    */
/* The following statements contradict and supercede related information        */
/* about font family numbers in the 1988 Apple IIGS Toolbox Reference Manual:   */
/* Always determine the famNum part of a font id using font manager calls. 	    */
/* The family number for a font family known by a particular name may differ    */
/* each time the font manager is started!									    */
/* Do not use statements like 'MyFontID.famNum = 22;' to select the Courier     */
/* font family, for instance.												    */
/*																			    */
/* Use GetFamNum to determine the font family number for a particularly named   */      							  
/* font family. See the AddApFont routine for an example of calling GetFamNum.  */
/*																			    */
/* ***************************************************************************  */


#include <types.h>     
#include <prodos.h>                             
#include <misctool.h>                            
#include <quickdraw.h>                         
#include <window.h>
#include <memory.h>
#include <dialog.h>
#include <menu.h>
#include <control.h>
#include <desk.h>
#include <event.h>
#include <lineedit.h>
#include <misctool.h>
#include <locator.h>
#include <Stdfile.h>
#include <qdaux.h>
#include <print.h>
#include <font.h>
#include <intmath.h>
#include <scrap.h>
#include <texttool.h>
#include <list.h>
#include <Math.h>
#include <Resources.h>

extern int _toolErr;

#if productionVersion
#define initPtrCheck(x);
#define closePtrCheck();
#define zapLocals();
#define checkForHit();
#endif

/* ************************************************************************* */
/*																			 */
/* Standard Global data here												 */
/*																			 */
/* ************************************************************************* */

#define ScreenWidth 640

int MyID;
int ThisMode = mode640;                           /* init mode = 640 */

Ref initRef;        				/* This holds the reference to the startstop record */

int QuitFlag;
QuitRec QuitParms = {
		NULL,						/* Pathname of the next application */
		0};							/* Quit Flags */
		
WmTaskRec EventRec = {				/* Data block to hold event records */
		0,
		0L,
		0L,
		0,0,
		0,
		0L,
		0x0000FFFFL};				/* Let TaskMaster do it all! */


/* ************************************************************************* */
/*																			 */
/* ErrorCheck - Called from InitTools to check for startup errors.			 */
/*																			 */
/* ************************************************************************* */

char DeathMsg[] = "Fatal error $0000 has occurred at 0000. Press any key to exit:";

void ErrorCheck(where)
	int where;
{
	if (_toolErr) {
		Int2Hex(_toolErr,DeathMsg + 13 ,4);	/* Stick error # into a string */
		Int2Hex(where,DeathMsg + 34 ,4);		/* Stick location # into a string */
		GrafOff();							/* Turn off Super Hires */
		WriteCString(DeathMsg);				/* Print the error message */
		ReadChar(0);							/* Pause for a character */
		ShutDownTools(refIsHandle, initRef);	/* Let the toolbox shutdown the tools. */
		exit(1);								/* exit */
	}
}


/* ************************************************************************* */
/*																			 */
/* ApErrorCheck - Called to check for tool and other miscellaneous errors.	 */
/*																			 */
/* ************************************************************************* */

char ApErrMsg[] = "Appl. error $0000 has occurred at 0000. Press any key to exit:";

void ApErrorCheck(where)
int where;
{
	int My_toolErr;

	zapLocals();

	My_toolErr = _toolErr;
	if (_toolErr) {
		Int2Hex(_toolErr,ApErrMsg + 13 ,4);		/* Stick error # into a string */
		Int2Hex(where,ApErrMsg + 34 ,4);		/* Stick location # into a string */
		GrafOff();								/* Turn off Super Hires */
		WriteCString(ApErrMsg);					/* Print the error message */
		ReadChar(0);							/* Pause for a character */
		if ((My_toolErr & 0xFF00) == 0x200 ) 	/* abort sample application if memory manager error */
		{
			ShutDownTools(refIsHandle, initRef);	/* Let the toolbox shutdown the tools. */
			exit(1);								/* exit */
		}
		GrafOn();								/* Turn on Super Hires */
	}
}


/* ************************************************************************* */
/*																			 */
/* Application specific data appears here									 */
/*																			 */
/* ************************************************************************* */

#define TLStartErr						1
#define ShowFontInstallFontErr			2
#define MyDrawLoadSysFontErr			3
#define MyDraw2LoadSysFontErr			4
#define ApFontNewHandleErr				5
#define ApFontAddFamilyErr				6
#define ApFontAddFontVarErr				7
#define ApFontInstallFontErr1			8
#define ApFontInstallFontErr2			9
#define InvFontLoadSysFontErr			10
#define InvFontLoadFontErr				11
#define InvFontSetPurgeStatErr			12
#define DispNewFontInstallFontErr		13
#define FindAltSysFontFMSetSysFontErr	14
#define MenuFontChooseErr				15
#define InitAppInstallFontErr			16

#define NumExampleLines 5					/* used by ShowFont	*/
#define NumStrikeLines 8					/* used by ShowFont	*/
#define ScalingOn	0						/* used calling InstallFont */
#define IllegalFontFamNum 0xffff			/* used by AddApFont */
#define Hex 16								/* used calling DrawCatInt2Base */
#define Dec 10								/* used calling DrawCatInt2Base */
#define signednum true						/* used calling DrawCatInt2Base */
#define loadfonts true						/* used calling InventoryFonts */
#define dontloadfonts false					/* used calling InventoryFonts */
#define douncheck true						/* used calling FontCheckMItem */
#define omituncheck false					/* used calling FontCheckMItem */

#define CatalogWindowID		0x1000L
#define StrikeWindowID		0x1001L

/* *************************************************************************  */
/*
/* This sample program uses menu items that start at 250. The Edit menu items */
/* if they were present would be entered first to conform to the menu manager */
/* documentation. The About and Quit menu items use IDs 256 and 257 as a      */
/* convention.					 											  */
/*																			  */
/* *************************************************************************  */

#define AppleMenuID			0x1100
#define AboutID				0x1101
#define FileMenuID			0x1200
#define OpenCatWindowID		0x1201
#define OpenStrikeWindowID	0x1202
#define QuitID				257
#define FontMenuID			0x1300
#define ChooseID			0x1301


int CurrHeight,LineCounter;

char Line0[30] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,  /*   Namelength + 1  */
                  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0}; /* + 4 for size info */
char Line1[] = "\0";
char Line2[] = "\54The quick brown fox jumps over the lazy dog.";
char Line3[] = "\53She sells sea shells down by the sea shore.";
char Line4[] = "\0";

char FontStrikeLine[] = {32,
                 0, 1, 2, 3, 4, 5, 6, 7, 8, 9,10,11,12,13,14,15,
                16,17,18,19,20,21,22,23,24,25,25,27,28,28,30,31,0};


char *LineTable[] = {Line0,Line1,Line2,Line3,Line4};

char __dataBank;

ptr myApFontNamePtr;

Point CurrPos;

FontID curSysID;

FontID FontToShow;

FontID ChosenFont;

int ChosenSize;

int ChosenStyle;

int checkedID;

int MyFamSpecBits;

struct FontStatRecList{
	int numFontsAvail;
	int numFamsAvail;
	FontStatRec FontStatRecTable[200];
	} FontsAvail;

int linetodraw;								/* shared by each window's content def proc */

GrafPortPtr CatalogWindow;                   /* current window handle  */
GrafPortPtr StrikeWindow;                    /* current window handle  */


/* ************************************************************************* */
/* Our sample programs use menus that start at 250. The Edit menu items are  */
/* entered first to conform to the menu manager documentation. The About     */
/* and Quit menu items use IDs 256 and 257 as a convention.					 */
/*																			 */
/* ************************************************************************* */

int numFontMenuItems; /* set to number of items in font menu, after FixFontMenu is called */


/* ************************************************************************* */
/*																			 */
/* This is the core of the program. It simply calls all the other routines   */
/* in order.																 */
/*																			 */
/* ************************************************************************* */

main()
{
	/* Startup the tools using the new toolbox call */
	MyID = MMStartUp();
	initRef = StartUpTools(MyID, refIsResource, 0x0001L);

	initPtrCheck(_ownerid);

	ApErrorCheck(TLStartErr);	/* Is StartUpTools OK? */
	if (!_toolErr) {
		InitApp();				/* initialize application specific stuff */
		EventLoop();			/* do your stuff */
	}
	ShutDownTools(refIsHandle, initRef);	/* Let the toolbox shutdown the tools. */

	closePtrCheck();
}


/* ********************************************************************************* */
/*																			 		 */
/* Assembly Procedure SaveDB:														 */
/*																			 		 */
/* Decsription: saves data bank register and changes to data bank for globals 		 */
/*																			 		 */
/* Inputs: 	none						 						 					 */
/*																			 		 */
/* Outputs: data bank register contents												 */
/*																			 		 */
/* ********************************************************************************* */

asm(SaveDB) {
             phb
             lda  #^__dataBank   ; high word of anything in ~globals will do
             pha
             plb
             pla

             rtl
}

/* ********************************************************************************* */
/*																			 		 */
/* Assembly Procedure RestoreDB:													 */
/*																			 		 */
/* Decsription: restores data bank register								 			 */
/*																			 		 */
/* Inputs: 	integer value to set data bank register to 						 		 */
/*																			 		 */
/* Outputs: changed data bank register												 */
/*																			 		 */
/* ********************************************************************************* */

asm(RestoreDB) {
             lda  4,s
             pha
             plb
             plb
             rtl
}

/* ********************************************************************************* */
/*																			 		 */
/* Procedure CalcLineHeight:														 */
/*																			 		 */
/* Decsription: calculates heigth of the text line(s) to be drawn		 			 */
/*																			 		 */
/* Inputs: 	none 						 											 */
/*																			 		 */
/* Outputs: heigth of the line(s) to be drawn										 */
/*																			 		 */
/* ********************************************************************************* */

CalcLineHeight()
{
	FontInfoRecord CurrFont;
	int LineHeight;
	
	zapLocals();

	GetFontInfo(&CurrFont);									/* prepare to compute height using */
															/* QuickDraw II GetFontInfo call */
															/* i.e. # pixels between lines drawn */
	LineHeight = CurrFont.ascent + CurrFont.descent + CurrFont.leading;
	return(LineHeight);
}


/* ********************************************************************************* */
/*																			 		 */
/* Procedure ShowFont:																 */
/*																			 		 */
/* Decsription: displays fontstrike, excerpted from HodgePodge example program		 */
/*																			 		 */
/* Inputs: 	pointer to fontID 						 								 */
/*																			 		 */
/* Outputs: none																	 */
/*																			 		 */
/* ********************************************************************************* */

ShowFont(fontId)
FontID fontId;
{

#define columnone	 	4

	int stringpos;											/* loops through ASCII values 	*/
   
	zapLocals();

	InstallFont(fontId.fidLong, ScalingOn);				/* Install with scaling enabled */
	ApErrorCheck(ShowFontInstallFontErr);					/* Is InstallFont OK? */
	CurrHeight = CalcLineHeight();							/* calculate # pixels between lines */

	GetFamInfo(fontId.fidRec.famNum, Line0);          		/* set up font name, ignore result */
	Int2Dec(fontId.fidRec.fontSize,              			/* set up font size after font name */
			   (Line0)+Line0[0]+1,       					/* pointer to end */
			   4,                                 			/* length of result */
			   0);                                         	/* not signed */
	Line0[0] +=4;                       					/* new length */
	linetodraw = linetodraw + 15;
   
	for (LineCounter = 0;LineCounter < NumExampleLines;LineCounter++) {
		linetodraw = linetodraw + CurrHeight;
		MoveTo(columnone,linetodraw);            			/* move to beginning of line to draw */
		DrawString(LineTable[LineCounter]);
	}
	for (LineCounter = 0;LineCounter < NumStrikeLines;LineCounter++) {
		linetodraw = linetodraw + CurrHeight;
		MoveTo(columnone,linetodraw);            			/* set position to Draw */
		for (stringpos = 1;stringpos < 33;stringpos++) {
			FontStrikeLine[stringpos] = (char) (LineCounter*32+stringpos-1);
		}
		DrawString(FontStrikeLine);
	}
	linetodraw = 0;
	MoveTo(columnone,0);
}


/* ********************************************************************************* */
/*																			 		 */
/* Procedure DrawCatInt2Base:														 */
/*																			 		 */
/* Decsription: displays number and string						 					 */
/*																			 		 */
/* Inputs: 	number to draw, base of number, number of places to display 			 */
/*			signed number indicator, append string							 		 */
/*																			 		 */
/* Outputs: none																	 */
/*																			 		 */
/* ********************************************************************************* */

DrawCatInt2Base(num,base,numplaces,signedval,catstr)

int num;												/* number to draw */
int base;												/* base of number */
int numplaces;											/* number of places to display */
int signedval;											/* signed number indicator */
char catstr[80];										/* append string */

{
	char strtarg[80];									/* convert number into this string */
	ptr catstrptr;										/* ptr to append string */
	ptr strtargptr;										/* ptr to convert string */
	
	zapLocals();

	catstrptr  = catstr;								/* set the pointers */
	strtargptr = strtarg;
	switch(base) {										/* do the conversion */
		case Dec:
			Int2Dec(num,strtargptr,numplaces,signedval);
			break;
		case Hex:
			Int2Hex(num,strtargptr,numplaces);
			break;
		default:
			/* error */
			break;
	}
	
	strtarg[numplaces]= '\0';								/* terminate the string */
	strcat(strtargptr,catstrptr);							/* append designated string */
	DrawCString(strtarg);									/* display result */
}


/* ********************************************************************************* */
/*																			 		 */
/* Procedure MyDraw:																 */
/*																			 		 */
/* Decsription: window content definition procedure				 					 */
/*																			 		 */
/* Inputs: 	none										 	 			 			 */
/*																			 		 */
/* Outputs: none																	 */
/*																			 		 */
/* ********************************************************************************* */

MyDraw()												/* window content definition procedure */	

{
	/* #define columnone	 	4 */
	#define famNumbercolumn		34*6+columnone
	#define fontSizecolumn		54*6+columnone
	#define fontStylecolumn		68*6+columnone
	char stritem1[80],stritem2[80],stritem3[80],stritem4[80];		/* string declarations */
	char stritem5[80],stritem6[80],stritem7[80];
	ptr strptr1,strptr2,strptr3,strptr4;							/* string ptr declarations */
	char TableTitle[80];											/* title string */
	int eachfontavail;												/* index into font catalog */
	int preservefontavail;											/* holds index temporarily */
	FontStatRecPtr MyFontStatRecPtr;								/* ptr to font status record */
	FontStatRecPtr PreserveFStatRecPtr;								/* holds ptr temporarily */
	int curfamily;													/* tracks font families */
	int MyFamStatBits;												/* FamStatBits */
	int dbr;														/* data bank register value */
	
	zapLocals();

    dbr = SaveDB();													/* make the data bank reg. */
																	/* point to globals, save it */
	strptr1 = stritem1;
	strptr2 = stritem2;
	strptr3 = stritem3;
	strptr4 = stritem4;

	strcpy(TableTitle,"\pReport of available fonts:");
	LoadSysFont();													/* use sys font */
	ApErrorCheck(MyDrawLoadSysFontErr);								/* Is LoadSysFont OK? */
	strcpy(stritem5," fonts available, ");							/* append strings */
	strcpy(stritem6," families");
	strcpy(stritem7,"");
	MoveTo(columnone,50);
	DrawString(TableTitle);
	DrawCatInt2Base(FontsAvail.numFontsAvail,Dec,4,signednum," fonts available, ");
	DrawCatInt2Base(FontsAvail.numFamsAvail,Dec,4,signednum,stritem6);
	
	curfamily = 0xffff;												/* init current family */
	strcpy(stritem1," ");											/* init strings */
	strcpy(stritem2," ");
	strcpy(stritem3," ");
	strcpy(stritem4," ");
	eachfontavail = 0;
	linetodraw = 50;
	/*               012345678901234567890123456789012345678901234567890123456789012345678901234567 */
	strcpy(stritem2,"Font family name                  family number       font size     font style");
	linetodraw = linetodraw+30;
	MoveTo(columnone,linetodraw);
	DrawCString(stritem2);
	do {																/* loop through fonts */
		MyFontStatRecPtr = &FontsAvail.FontStatRecTable[eachfontavail];	/* set pointer */
		strcpy(stritem2," ");											/* init strings */
		strcpy(stritem3," ");
		strcpy(stritem4," ");
		linetodraw = linetodraw+10;
		if (MyFontStatRecPtr->resultID.fidRec.famNum != curfamily) {	/* next family? */
																		/* yes- display */
			MyFamStatBits = GetFamInfo(MyFontStatRecPtr->resultID.fidRec.famNum,strptr1);
			MoveTo(columnone,linetodraw);
			DrawString(stritem1);
			MoveTo(famNumbercolumn,linetodraw);
			DrawCatInt2Base(MyFontStatRecPtr->resultID.fidRec.famNum,Hex,4,signednum,stritem7);
		}
																	   /* display size and style */
		curfamily = MyFontStatRecPtr->resultID.fidRec.famNum;
		MoveTo(fontSizecolumn,linetodraw);
		DrawCatInt2Base(MyFontStatRecPtr->resultID.fidRec.fontSize,Dec,5,signednum,stritem7);
		MoveTo(fontStylecolumn,linetodraw);
		DrawCatInt2Base(MyFontStatRecPtr->resultID.fidRec.fontStyle,Dec,5,signednum,stritem7);
		
		++eachfontavail;
	}
	while (eachfontavail < FontsAvail.numFontsAvail);
    RestoreDB(dbr);											/* restore data bank reg. for return */
															/* to toolbox */
}


/* ********************************************************************************* */
/*																			 		 */
/* Procedure MyDraw2:																 */
/*																			 		 */
/* Decsription: window content definition procedure				 					 */
/*																			 		 */
/* Inputs: 	none										 	 			 			 */
/*																			 		 */
/* Outputs: none																	 */
/*																			 		 */
/* ********************************************************************************* */

MyDraw2()												/* window content definition procedure */	

{
	int dbr;												/* data bank register value */
	
	zapLocals();

	dbr = SaveDB();											/* make the data bank reg. */
															/* point to globals, save it */
	linetodraw = 0;
															/* all families, base & non-base */
	ShowFont(FontToShow.fidLong);							/* display font strike */
	LoadSysFont();
	ApErrorCheck(MyDraw2LoadSysFontErr);					/* Is LoadSysFont OK? */
	RestoreDB(dbr);											/* restore data bank reg. for return */
															/* to toolbox */
}


/* ********************************************************************************* */
/*																			 		 */
/* Procedure FontCheckMItem															 */
/*																			 		 */
/* Decsription: This procedure checks specified font item in the menu bar and		 */
/*				unchecks previously checked item, if any							 */
/*																			 		 */
/* Inputs: 	pointer to FontID to be checked in menu bar, 						 	 */
/*			flag indicating whether previously checked item exists.				 	 */
/*																			 		 */
/* Outputs: none																 	 */
/*																			 		 */
/* ********************************************************************************* */

FontCheckMItem(FontToCheck, uncheck)
FontID FontToCheck;											/* pointer to FontID to be checked */
int uncheck;												/* flag if prev checked item exists */
{
	int prevcheckedID;										/* menu id */
	
	zapLocals();

	prevcheckedID = checkedID;
	if (uncheck = true) CheckMItem(false,prevcheckedID);		/* uncheck menu item */
	checkedID = FamNum2ItemID(FontToCheck.fidRec.famNum);
	CheckMItem(true,checkedID);									/* check menu item */
}


/* ********************************************************************************* */
/*																			 		 */
/* Procedure AddApFont:																 */
/*																			 		 */
/* Decsription: copies system font into application created font,		 			 */
/*				generates unique name and number of font family						 */
/*																			 		 */
/* Inputs: 	none			 	 										 			 */
/*																			 		 */
/* Outputs: none																	 */
/*																			 		 */
/* ********************************************************************************* */

AddApFont()
{
	#define FontFamNameOverflow 27	
	#define UnusedFontFamNum 0x80
	FontID MyFont;											/* fontIDs proper and long form */
	char stritem1[27],stritem2[27];							/* misc. strings and ptrs to strings */
	ptr strptr1,strptr2;
	char myApFontName[27],fontname[80];						/* Font name strings */
	ptr fontnameptr;										/* ptr to Font name string */
	FontHndl MyApFontHndl,MySysFontHandle;					/* Handles to font */
	int *MyApFontPatchPtr;									/* ptr into font */
	long MyApFontSize;										/* size of font */
	int MyFontSpecBits,MyFamStatBits,MyFamSpecBits;			/* misc. Font Bits */
	int MyApFamNum;											/* font family number */
	int nthfontname,numsuffixplaces,numleadingspaces;		/* used to create unique font name */
	FontStatRec MyFontStatRec;								/* Font status record and StatRecPtr */
	FontStatRecPtr MyFontStatRecPtr;

	zapLocals();

	strptr1 = stritem1;
	strptr2 = stritem2;

	strcpy(stritem1,"                          ");
	strcpy(myApFontName,"ApFont");
	
	MyFont.fidLong = FMGetCurFID();							/* install 16 point version of font */
	
	MyFont.fidRec.fontSize = 16;
	InstallFont(MyFont.fidLong, ScalingOn);					/* install */
	ApErrorCheck(ApFontInstallFontErr1);					/* Is InstallFont OK? */
	
	MySysFontHandle = GetFont();							/* prepare to duplicate the font */
	MyApFontSize = GetHandleSize(MySysFontHandle);			/* determine size of font */
															/* make room for duplicate */
	MyApFontHndl = (FontHndl) NewHandle(MyApFontSize,MyID,0,0L);	/* make room for duplicate */
	HLock(MyApFontHndl);
	ApErrorCheck(ApFontNewHandleErr);						/* Is NewHandle OK? */
	
	if (!_toolErr) {
	
		HandToHand(MySysFontHandle,(Handle) MyApFontHndl,MyApFontSize);
		MyApFamNum = 0xfffe;					/* prepare to validate font number not used */

		while (!(GetFamInfo(MyApFamNum,strptr1) & notFoundBit)) --MyApFamNum;

		myApFontNamePtr = myApFontName;			/* prepare to validate font name not used */
		strcpy(stritem1,myApFontName);			/* init temp font name */ 
		nthfontname = 0;						/* may be used to suffix font name */
												/* loop until illegal or exhausted */

		while (
			(GetFamNum(strptr1) != IllegalFontFamNum) && (strlen(stritem1) != FontFamNameOverflow)
		) {
			++nthfontname;									/* bump the suffix value */
			numsuffixplaces = (int) log10(nthfontname) + 1;	/* calc # digits in suffix */
															/* make a string of suffix */
			Int2Dec(nthfontname,strptr2,numsuffixplaces,true);
			stritem2[numsuffixplaces]= '\0';				/* terminate suffix string */
			strcpy(stritem1,myApFontName);					/* reinit temp font name */
			strcat(strptr1,strptr2);						/* append suffix */
		}

		c2pstr(myApFontNamePtr);
		AddFamily(MyApFamNum,myApFontNamePtr);
		ApErrorCheck(ApFontAddFamilyErr);						/* Is AddFamily OK? */

												/* patch the font # of the font record */
		MyApFontPatchPtr =  (int *) (((char *) *MyApFontHndl) + 2);
		*MyApFontPatchPtr = MyApFamNum;
		AddFontVar(MyApFontHndl,0);
		ApErrorCheck(ApFontAddFontVarErr);						/* Is AddFontVar OK? */
		HUnlock(MyApFontHndl);
		MyFont.fidRec.famNum = MyApFamNum;
		InstallFont(MyFont.fidLong, ScalingOn);
		ApErrorCheck(ApFontInstallFontErr2);	/* Is InstallFont OK? */
	}
}


/* ********************************************************************************* */
/*																			 		 */
/* Procedure InventoryFonts:														 */
/*																			 		 */
/* Decsription: produces font catalog consisting of number of families 		 		 */
/*				(base & non-base), generates unique name and number of font family.	 */
/*				assumes LoadFont and FindFontStats group fonts by size in increasing */
/*				order within each family and by style in increasing order within	 */
/*				each size. If fonts are to be loaded, they are also set to be 		 */
/*				purgeable.			 										 		 */
/*																			 		 */
/* Inputs: 	flag whether to load fonts as they are being inventoried				 */
/*																			 		 */
/* Outputs: none																	 */
/*																			 		 */
/* ********************************************************************************* */

InventoryFonts(loadthefonts)
int loadthefonts;

{
	FontID SelectFontID;							/* fontIDs proper and long form */
	char stritem1[27],stritem2[27];					/* misc. strings and ptrs to strings */
	ptr strptr1,strptr2;
	char fontname[80];
	ptr fontnameptr;
															/* used to loop through fonts*/
	int eachfamavail,numfontsinfam,eachfontinfam,eachfontavail;
	int cursize,numsizesinfont,numstylesinsize;				/* used to loop through fonts */
															/* misc. font values*/
	int MyFontSpecBits,MyOtherFontSpecBits,MyFamStatBits,MyFamSpecBits; 
	FontStatRec MyFontStatRec;								/* Font status record and StatRecPtr */
	FontStatRecPtr MyFontStatRecPtr;

	zapLocals();

	strptr1 = stritem1;
	strptr2 = stritem2;
	
	LoadSysFont();
	ApErrorCheck(InvFontLoadSysFontErr);						/* Is LoadSysFont OK? */
	
	MyFamSpecBits = 0;
	FontsAvail.numFamsAvail = CountFamilies(MyFamSpecBits);		/* determine no. of families */
	MyFontSpecBits = realOnlyBit | anyFamBit | anyStyleBit | anySizeBit;   /* all real fonts */
	SelectFontID.fidRec.famNum = 0;
	SelectFontID.fidRec.fontStyle = 0xff;								/* (any style) */
	SelectFontID.fidRec.fontSize = 1;
	FontsAvail.numFontsAvail = CountFonts(SelectFontID.fidLong, MyFontSpecBits);
																/* determine # of fonts */

	eachfamavail = 0;											/* init family counter */
	eachfontavail = 0;											/* init font counter */
	fontnameptr = fontname;										/* set up ptr to font name string*/
	MyFontSpecBits = realOnlyBit | anyStyleBit | anySizeBit;	/* all real fonts in each family */

	do {											/*loop through font family list */
		++eachfamavail;								/* increment family counter */

													/* determine number of nth family */
		SelectFontID.fidRec.famNum = FindFamily(MyFamSpecBits,eachfamavail,fontnameptr);

		numfontsinfam = CountFonts(SelectFontID.fidLong, MyFontSpecBits);
													/* determine # fonts in that family */

		eachfontinfam = 0;							/* init fonts in family counter */
		numsizesinfont = 0;							/* init sizes in family counter */
		cursize = 0;								/* init current size */

		do {										/* loop through the fonts in this family */
			++eachfontinfam;					/* increment fonts in family counter */

												/* set pointer to nth font status record */
			MyFontStatRecPtr = &FontsAvail.FontStatRecTable[eachfontavail];

			switch(loadthefonts) {
				case loadfonts:
					/* assume LoadFont and FindFontStats group fonts by size in increasing */
					/* order within each family and by style in increasing order within	 */
					/* each size */

					/* there was a bug in the status record returned by LoadFont prior to */
					/* GS/OS, substitute FindFontStats,InstallFont */
					/* in place of LoadFont if and when running under pre-GS/OS system */
					/* to return correct status record */

					/* make current and */
					/* determine stats of nth font in family */
					LoadFont(SelectFontID.fidLong, MyFontSpecBits,eachfontinfam,MyFontStatRecPtr);
					ApErrorCheck(InvFontLoadFontErr);	/* Is InstallFont OK? */

					/* prepare to set purge bit, at discretion of application */
					/* check first that not attempting to set purge bit of system font */
					if (MyFontStatRecPtr->resultID.fidLong != curSysID.fidLong)
						MyFontStatRecPtr->resultStats = 
							MyFontStatRecPtr->resultStats | purgeBit;

					/* set purge bit */
					/* !!!! */
					/* When fonts are to be marked purgeable using SetPurgeStat, the id of */
					/* the font to purge should be that returned from a font manager call, */
					/* such as FMGetCurFID, FindFontStats, or LoadFont. */

					SetPurgeStat(MyFontStatRecPtr->resultID,MyFontStatRecPtr->resultStats);

													/* Is SetPurgeStat OK? */
					ApErrorCheck(InvFontSetPurgeStatErr);
					break;
				
				case dontloadfonts:
													/* determine stats of nth font in family */
					FindFontStats(SelectFontID.fidLong, MyFontSpecBits,
					              eachfontinfam, MyFontStatRecPtr);
					break;
					
				default:
					/*error */
					break;
			}
				
											/* does the font size equal previous font? */
			if (MyFontStatRecPtr->resultID.fidRec.fontSize != cursize)
			{
				++numsizesinfont;			/* no- then increment number of sizes/family*/
				numstylesinsize = 0;		/*     and set to zero number of styles/size*/
			}
											/* store current font size for comparison */
			cursize = MyFontStatRecPtr->resultID.fidRec.fontSize;
			++numstylesinsize;				/* increment number of styles/size */
			++eachfontavail;				/* increment fonts counter */
		}
		while (eachfontinfam < numfontsinfam);
	}
	while (eachfamavail < FontsAvail.numFamsAvail);
}


/* ********************************************************************************* */
/*																			 		 */
/* Procedure FindAltSysFont															 */
/*																			 		 */
/* Decsription: resets system font to first other font cataloged with same size		 */
/*																			 		 */
/* Inputs: 	none																	 */
/*																			 		 */
/* Outputs: none																 	 */
/*																			 		 */
/* ********************************************************************************* */

FindAltSysFont()
{
	int cursize,eachfontavail;				/* loop variables */
	FontStatRecPtr MyFontStatRecPtr;		/* points to font status record */
	
	zapLocals();

	cursize = curSysID.fidRec.fontSize;
	eachfontavail = 0;							/* init font counter */
	do {										/* loop through the font catalog */
												/* set pointer to nth font status record */
		MyFontStatRecPtr = &FontsAvail.FontStatRecTable[eachfontavail];
		++eachfontavail;
	}
	while ((eachfontavail<FontsAvail.numFontsAvail) && 
		((MyFontStatRecPtr->resultID.fidRec.fontSize != cursize) || 
		 (MyFontStatRecPtr->resultID.fidLong == curSysID.fidLong)));

										/* change system font if found other font of same size */
	if ((MyFontStatRecPtr->resultID.fidLong != curSysID.fidLong) &&
		(MyFontStatRecPtr->resultID.fidRec.fontSize == cursize)) {
		FMSetSysFont(MyFontStatRecPtr->resultID);
												/* Is FMSetSysFontErr OK? */
		ApErrorCheck(FindAltSysFontFMSetSysFontErr);
		curSysID.fidLong = FMGetSysFID();
	}
}


/* ********************************************************************************* */
/*																			 		 */
/* Procedure DispNewFont:														 	 */
/*																			 		 */
/* Decsription: resizes window data and forces redisplay of window contents 		 */
/*																			 		 */
/* Inputs: 	none																	 */
/*																			 		 */
/* Outputs: none																	 */
/*																			 		 */
/* ********************************************************************************* */

DispNewFont()
{
	int drawheight;

	zapLocals();

	InstallFont(FontToShow.fidLong, ScalingOn);
	ApErrorCheck(DispNewFontInstallFontErr);					/* Is InstallFont OK? */
    CurrHeight = CalcLineHeight();								/* calc # pixels between lines */
	drawheight = 40 + (13*CurrHeight) + 20;
	SetDataSize(ScreenWidth,drawheight,StrikeWindow);
	SetPort(StrikeWindow);										/* insure correct port */
	EraseRect(&(*StrikeWindow).portRect);						/* erase the port */
	InvalRect(&(*StrikeWindow).portRect);						/* and cause it to be redrawn */
}


/* ********************************************************************************* */
/*																			 		 */
/* Procedure MenuFontSelect:														 */
/*																			 		 */
/* Decsription: checks the item in menu bar and initiates display of font strike 	 */
/*																			 		 */
/* Inputs: 	user selected font menu item											 */
/*																			 		 */
/* Outputs: none																	 */
/*																			 		 */
/* ********************************************************************************* */

MenuFontSelect()
{
	int theMenuID;									/* gets user selected font menu item */
	
	zapLocals();

	theMenuID = (int) (EventRec.wmTaskData);				/* pick up menuID */
															/* determine font ID */
	FontToShow.fidRec.famNum = ItemID2FamNum(theMenuID);	/* determine family # */
	FontToShow.fidRec.fontSize = ChosenSize;				/* set pt. font */
	FontToShow.fidRec.fontStyle = ChosenStyle;				/* set style	*/
	FontCheckMItem(FontToShow.fidLong, douncheck);			/* check font in menu */
	DispNewFont();											/* display font strike */														
}


/* ********************************************************************************* */
/*																			 		 */
/* Procedure MenuFontChoose:														 */
/*																			 		 */
/* Decsription:	Invokes choose font dialog, checks the user specified item in	 	 */
/*				menu bar, and initiates display of font strike				 		 */
/*																			 		 */
/* Inputs: 	user selection of 'choose' from font menu								 */
/*																			 		 */
/* Outputs: none																	 */
/*																			 		 */
/* ********************************************************************************* */

MenuFontChoose()
{
	GrafPortPtr preserveport;

	zapLocals();

	preserveport = GetPort();
	SetPort(StrikeWindow);									/* operate on strike port */
	
	ChosenFont.fidLong = ChooseFont(FontToShow.fidLong, 0);	/* user font choice */
	ApErrorCheck(MenuFontChooseErr);						/* Is ChooseFont OK? */
	if (ChosenFont.fidLong) {
		FontToShow.fidLong = ChosenFont.fidLong;
		ChosenSize = FontToShow.fidRec.fontSize;			/* store font size chosen */
		ChosenStyle = FontToShow.fidRec.fontStyle;			/* store font style chosen */
		FontCheckMItem(FontToShow.fidLong, douncheck);		/* check font in menu */
		DispNewFont();										/* display font strike */								
	}
	
	SetPort(preserveport);									/* restore port */
}


/* ************************************************************************* */
/*																			 */
/* Procedure OpenCatalogWindow 												 */
/*																			 */
/* Decsription:	This routine creates and shows the catalog window and		 */
/*				disables the menu item for opening the catalog window		 */
/*																			 */
/* ************************************************************************* */

OpenCatalogWindow()
{
    CatalogWindow = NewWindow2(NULL, NULL, MyDraw, NULL, 2,
		CatalogWindowID, rWindParam1);	 							/* create and show window */
	SetPort(CatalogWindow);
	ShowWindow(CatalogWindow);
	DisableMItem(OpenCatWindowID);									/* disable menu item */
}


/* ************************************************************************* */
/*																			 */
/* Procedure OpenStrikeWindow 												 */
/*																			 */
/* Decsription:	This routine creates and shows the font strike  window and	 */
/*				disables the menu item for opening the font strike window	 */
/*																			 */
/* ************************************************************************* */

OpenStrikeWindow()
{
    StrikeWindow = NewWindow2(NULL, NULL, MyDraw2, NULL, 2,
		StrikeWindowID, rWindParam1);								/* create and show window */
	SetPort(StrikeWindow);
	ShowWindow(StrikeWindow);
	SetMenuFlag(enableMenu,FontMenuID);								/* enable the font menu */
	HiliteMenu(FALSE,FontMenuID);									/* force it (only) to redraw */
	DisableMItem(OpenStrikeWindowID);								/* disable menu item */
}


/* ************************************************************************* */
/*																			 */
/* InitApp - This routine is called once after the tools are started. This   */
/* is where you would create your menus, and other objects your program will */
/* need at the very start.													 */
/*																			 */
/* ************************************************************************* */

InitApp()
{
	int menuheight;									/* height of menu bar */
	int maxfontmitems;								/* max # of fonts to list in menu */
	int menuversioncheck;							/* menu version less prototype bit */
	int i;											/* loop variable */

	zapLocals();

	WaitCursor();											/* inventory make take a while */
	ShowCursor();
	QuitFlag = 0;
	curSysID.fidLong = FMGetSysFID();						/* find and store the system fontID */
	AddApFont();
	InventoryFonts(loadfonts);
	FindAltSysFont();	  									/* make sys font any other 8 pt. font*/
	SetSysBar(NewMenuBar2(refIsResource, 0x0001L, NULL));
	SetMenuBar(NULL);
    
	FixAppleMenu(AppleMenuID);              				/* Add DAs to apple menu     */
	MyFamSpecBits = 0;
															/* append fonts to menu if */
															/* they'll fit, assume sys font */
															/* is 8 pt. font */
	menuheight = FixMenuBar();
	maxfontmitems = (int) ((200 - 2*menuheight) / menuheight);	
	menuversioncheck = MenuVersion() & 0x7fff;						/* strip the prototype bit */
	if ((FontsAvail.numFamsAvail <= maxfontmitems) || menuversioncheck >= 0x300)
		FixFontMenu(FontMenuID,ChooseID+1,MyFamSpecBits);			/* put fonts in font menu */
	CalcMenuSize(0,0,FontMenuID);									/* reset size of font menu */
	numFontMenuItems = CountMItems(FontMenuID);						/* used to handle font menu */
																	/* choices in DoMenuSelect */
	FontToShow.fidLong = FMGetSysFID();								/* get system font id */
	FontCheckMItem(FontToShow.fidLong, omituncheck);				/* check font in menu */						
	ChosenSize = FontToShow.fidRec.fontSize;						/* store font size chosen */
	ChosenStyle = FontToShow.fidRec.fontStyle;						/* store font style chosen */

	DrawMenuBar();
	
	OpenCatalogWindow();									/* create and show windows, */
	OpenStrikeWindow();										/* disable open window menu choices */
	SetPort(CatalogWindow);
	DispNewFont();											/* display font strike */													
	
	InitCursor();											/* restore cursor to normal */
}


/* ************************************************************************* */
/*																			 */
/* InGoAway - This routine gets called when the user clicks in the go away   */
/* box of a window. 														 */
/*																			 */
/* ************************************************************************* */

InGoAway()
{
	GrafPortPtr windowtoclose;

	zapLocals();

	windowtoclose = (GrafPortPtr) EventRec.wmTaskData;			/* close the window  */
	CloseWindow(windowtoclose);
	if (windowtoclose == CatalogWindow) 						/* enable open window menu opt. */
		EnableMItem(OpenCatWindowID);
	else if (windowtoclose == StrikeWindow) {
		SetMenuFlag(disableMenu,FontMenuID);					/* disable the font menu */
		HiliteMenu(FALSE,FontMenuID);							/* force it (only) to redraw */
		EnableMItem(OpenStrikeWindowID);
	}
}


/* ************************************************************************* */
/*																			 */
/* Ignore - called when an event occurs that you don't want/need to handle   */
/*																			 */
/* ************************************************************************* */

Ignore()
{
	/* Does nothing */
}


/* ************************************************************************* */
/*																			 */
/* DoAbout - Show the vanity box until the OK button is hit. Calls			 */
/* NoteAlert to display a dialog box with an icon and handle mouse clicks.   */
/*																			 */
/* ************************************************************************* */

DoAbout()
{
	AlertWindow(4,NULL, 0x0001L);
}


/* ************************************************************************* */
/*																			 */
/* DoUpdate - This routine is called if you have a window and the Event		 */
/* Manager notices that it needs to have some or all of it redrawn.			 */
/*																			 */
/* ************************************************************************* */

DoUpdate()
{ 
	/* no update */
} 


/* ************************************************************************* */
/*																			 */
/* DoQuit - Set the quit flag. This tells the Event loop to exit.			 */
/*																			 */
/* ************************************************************************* */

DoQuit()
{
	QuitFlag = -1;
}


DoMenuSelect()
{
    /* Procedure to handle all menu selections.  Examines the event.TaskData
    ** menu item ID word from TaskMaster (from Event Manager) and calls the
    ** appropriate routine.  While the routine is running the menu title is
    ** still highlighted.  After the routine returns, we unhilight the
    ** menu title.
    */
	
    unsigned int    menuNum, itemNum;

	zapLocals();

	menuNum = HiWord(EventRec.wmTaskData);
	itemNum = LoWord(EventRec.wmTaskData);

																	/* font menu font selection? */
	if ((itemNum > ChooseID) && (itemNum <= (ChooseID+numFontMenuItems)))
		MenuFontSelect();											/* yes- call this routine */
	else {
		switch (itemNum) {
			case AboutID:
				DoAbout();
				break;
			case OpenCatWindowID:
				OpenCatalogWindow();
				break;
			case OpenStrikeWindowID:
				OpenStrikeWindow();
				break;
			case QuitID:
				DoQuit();
				break;
			case ChooseID:
				MenuFontChoose();
				break;
		}
	}
    HiliteMenu(0, menuNum);     /* Unhighlight the menu title. */
}



/* ************************************************************************* */
/*																			 */
/* EventLoop - This routine is the heart of your application. It is repeat-  */
/* edly called and when the user selects/operates something, it dispatches   */
/* to a routine to do the proper thing.										 */
/*																			 */
/* ************************************************************************* */

EventLoop()
{
	word theTask;
	
	zapLocals();

	do {
		theTask = TaskMaster(everyEvent,&EventRec);	/* Get an event from TaskMaster */
		switch (theTask) {							/* switch on the event type. */
			case updateEvt:							/* Handle updates. */
				DoUpdate();
				break; 
			case wInMenuBar:						/* Handle selections in Special */
			case wInSpecial:						/* and normal menu item selects */
				DoMenuSelect();
				break;
			case wInGoAway:							/* Handle a click in the go away box */
				InGoAway();
				break;
		}
	} while (!QuitFlag);							/* Loop until "Quit" is selected */
}

